<?php
/**
 * Plugin Name: WC Bulk Variations Manager
 * Plugin URI: https://www.storeapps.org/product/bulk-variations-manager/
 * Description: Create WooCommerce Product Variations in bulk
 * Version: 3.1.6
 * Author: Ratnakar
 * Requires at least: 4.1
 * Tested up to: 4.9.9
 * WC requires at least: 2.5.0
 * WC tested up to: 3.5.3
 * Text Domain: wc-bulk-variations-manager
 * Domain Path: /languages/
 * License: GPLv2 or later
 * Copyright (c) 2013-2019 StoreApps All rights reserved.
 */

if ( !defined( 'ABSPATH' ) ) exit;

$active_plugins = (array) get_option( 'active_plugins', array() );

if ( is_multisite() ) {
    $active_plugins = array_merge( $active_plugins, get_site_option( 'active_sitewide_plugins', array() ) );
}

if ( ! ( in_array( 'woocommerce/woocommerce.php', $active_plugins ) || array_key_exists( 'woocommerce/woocommerce.php', $active_plugins ) ) ) return;

register_activation_hook ( __FILE__, 'bulk_variations_manager_activate' );

function bulk_variations_manager_activate() {
    if ( ! is_network_admin() && ! isset( $_GET['activate-multi'] ) ) {
        set_transient( '_bulk_variations_manager_activation_redirect', 1, 30 );            
    }    
}

include_once 'includes/class-bvm-admin-welcome.php';
require_once 'includes/wc-compat/version-2-5.php';
require_once 'includes/wc-compat/version-2-6.php';
require_once 'includes/wc-compat/version-3-0.php';

if ( !class_exists( 'SA_Bulk_Variations' ) ) {

	class SA_Bulk_Variations {

        static $text_domain;

		function __construct() {

            if ( in_array( 'smart-manager-for-wp-e-commerce/smart-manager.php', get_option( 'active_plugins' ) ) ) {
                add_action( 'admin_menu', array( $this, 'bvm_add_menu_access' ), 9 );
            } else {
                add_action( 'admin_menu', array( $this, 'woocommerce_variation_menu' ) );
            }

            add_action( 'admin_head', array( $this, 'bvm_admin_head' ) );

            add_action( 'init', array( $this, 'localize' ) );
            add_action( 'init', array( $this, 'bvm_include_classes' ) );
            add_action( 'admin_init', array( $this, 'activated' ) );
            add_filter( 'sa_is_page_for_notifications', array( $this, 'is_page_for_notifications' ), 10, 2 );

            add_action( 'wp_ajax_json_search_products_with_status', array( $this, 'json_search_products_with_status' ) );

            add_action( 'admin_footer', array( $this, 'bvm_support_ticket_content' ) );
            add_action( 'in_admin_footer', array( $this, 'add_social_links' ) );

            add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'plugin_action_links' ) );

            add_filter( 'the_category', array( $this, 'show_product_count' ) );
            add_filter( 'woocommerce_ajax_variation_threshold', array( $this, 'ajax_variation_threshold' ), 10, 2 );

            add_filter( 'sa_active_plugins_for_quick_help', array( $this, 'active_plugins_for_quick_help' ), 10, 2 );

            add_action( 'admin_notices', array( $this, 'needs_wc_25_above' ) );

		}

        /**
         * to handle WC compatibility related function call from appropriate class
         * 
         * @param $function_name string
         * @param $arguments array of arguments passed while calling $function_name
         * @return result of function call
         * 
         */
        public function __call( $function_name, $arguments = array() ) {

            if ( ! is_callable( 'SA_WC_Compatibility_3_0', $function_name ) ) return;

            if ( ! empty( $arguments ) ) {
                return call_user_func_array( 'SA_WC_Compatibility_3_0::'.$function_name, $arguments );
            } else {
                return call_user_func( 'SA_WC_Compatibility_3_0::'.$function_name );
            }

        }

        function bvm_add_menu_access() {
            global $wpdb, $current_user;
            $current_user = wp_get_current_user();
            if ( !isset( $current_user->roles[0] ) ) {
                $roles = array_values( $current_user->roles );
            } else {
                $roles = $current_user->roles;
            }
            $results = get_option( 'sm_' . $roles [0] . '_dashboard' );
            if ( ( is_array( $results ) && in_array( 'Products', $results, true ) ) || $current_user->roles [0] == 'administrator') {
                add_action( 'admin_menu', array( $this, 'woocommerce_variation_menu' ) );
            }
        }

		/**
         * Language loader
         */
        function localize() {

            $text_domains = array( 'wc-bulk-variations-manager', 'sa_bulk_variation' );        // For Backward Compatibility

            $plugin_dirname = dirname( plugin_basename(__FILE__) );

            foreach ( $text_domains as $text_domain ) {

                self::$text_domain = $text_domain;

                $locale = apply_filters( 'plugin_locale', get_locale(), self::$text_domain );

                $loaded = load_textdomain( self::$text_domain, WP_LANG_DIR . '/' . $plugin_dirname . '/' . self::$text_domain . '-' . $locale . '.mo' );

                if ( ! $loaded ) {
                    $loaded = load_plugin_textdomain( self::$text_domain, false, $plugin_dirname . '/languages' );
                }

                if ( $loaded ) {
                    break;
                }

            }

        }

        function bvm_include_classes() {
            global $bvm_operation_new, $bvm_operation_old;

        	include_once 'includes/class-bvm-operation-old.php';
        	include_once 'includes/class-bvm-operation-new.php';

        }

        public function activated() {
            $prefix = 'sa_bulk_variations';
            $is_check = get_option( $prefix . '_check_update', 'no' );
            if ( $is_check === 'no' ) {
                $response = wp_remote_get( 'https://www.storeapps.org/wp-admin/admin-ajax.php?action=check_update&plugin=bvm' );
                update_option( $prefix . '_check_update', 'yes' );
            }
        }

        /**
         * To determine whether to show notification on a page or not
         * 
         * @param bool $is_page
         * @param mixed $upgrader
         * 
         * @return bool $is_page
         */
        public function is_page_for_notifications( $is_page = false, $upgrader = null ) {

            $var_post_type = ( ! empty( $_GET['post_type'] ) ) ? $_GET['post_type'] : '';
            $var_page = ( ! empty( $_GET['page'] ) ) ? $_GET['page'] : '';

            if ( ! empty( $var_post_type ) && 'product' === $var_post_type && ! empty( $var_page ) && in_array( $var_page, array( 'woocommerce_variations', 'bvm-about', 'bvm-faqs' ) ) ) {
                return true;
            }
                
            return $is_page;
        }

        function json_search_products_with_status( $x = '' ) {

            check_ajax_referer( 'ajax-search-products-with-status', 'security' );
            
            $post_types = 'product';
            $post_status = ( !empty( $_GET['status'] ) ) ? unserialize( stripslashes( $_GET['status'] ) ) : 'any';

            $term = (string) stripslashes( $_GET['term'] );

            if (empty($term)) die();

            if ( is_numeric( $term ) ) {

                $args = array(
                    'post_type'         => $post_types,
                    'post_status'       => $post_status,
                    'posts_per_page'    => -1,
                    'post__in'          => array(0, $term),
                    'fields'            => 'ids'
                );

                $args2 = array(
                    'post_type'         => $post_types,
                    'post_status'       => $post_status,
                    'posts_per_page'    => -1,
                    'post_parent'       => $term,
                    'fields'            => 'ids'
                );

                $args3 = array(
                    'post_type'         => $post_types,
                    'post_status'       => $post_status,
                    'posts_per_page'    => -1,
                    'meta_query'        => array(
                        array(
                        'key'   => '_sku',
                        'value' => $term,
                        'compare' => 'LIKE'
                        )
                    ),
                    'fields'            => 'ids'
                );

                $posts = array_unique(array_merge( get_posts( $args ), get_posts( $args2 ), get_posts( $args3 ) ));

            } else {

                $args = array(
                    'post_type'         => $post_types,
                    'post_status'       => $post_status,
                    'posts_per_page'    => -1,
                    's'                 => $term,
                    'fields'            => 'ids'
                );

                $args2 = array(
                    'post_type'         => $post_types,
                    'post_status'       => $post_status,
                    'posts_per_page'    => -1,
                    'meta_query'        => array(
                        array(
                        'key'   => '_sku',
                        'value' => $term,
                        'compare' => 'LIKE'
                        )
                    ),
                    'fields'            => 'ids'
                );

                $posts = array_unique(array_merge( get_posts( $args ), get_posts( $args2 ) ));

            }

            $found_products = array();

            if ( $posts ) foreach ( $posts as $post ) {

                $product = wc_get_product( $post );

                $found_products[ $post ] = $product->get_formatted_name();

            }

            echo json_encode( $found_products );

            die();
        }

        function woocommerce_variation_menu() {
            global $wpdb, $current_user, $bvm_operation_new, $bvm_operation_old;
            
            if ( isset( $_GET['bvm_version'] ) && $_GET['bvm_version'] == 'old' ) {
                $bvm_operation = $bvm_operation_old;
            } else {
                $bvm_operation = $bvm_operation_new;

                if ( ! empty( $_GET['page'] ) && $_GET['page'] == 'woocommerce_variations' ) {

                    $is_permission_available = get_option( 'bvm_is_permission_available' );
                    $bvm_only_old_version = get_option( 'bvm_only_old_version' );

                    if ( $is_permission_available == 'no' || $bvm_only_old_version == 'yes' ) {
                        wp_safe_redirect( trailingslashit( admin_url() ) . 'edit.php?post_type=product&page=woocommerce_variations&bvm_version=old' );
                        exit;
                    }

                }
            }

            $bvm_operation = apply_filters( 'bvm_get_operation_object', $bvm_operation, $this );

            if( !wp_script_is('thickbox') ) {
                wp_enqueue_script('thickbox');
            }
            
            if( !wp_style_is('thickbox') ){
                 wp_enqueue_style('thickbox');
            }

            if ( (!current_user_can( 'edit_pages' )) && (is_plugin_active( 'woocommerce/woocommerce.php' )) ) {
                add_menu_page( __( 'WooCommerce Bulk Variations Manager', self::$text_domain ), __( 'Bulk Variations Manager Demo for WooCommerce', self::$text_domain ), 'read', 'woocommerce_variations', array( $bvm_operation, 'woocommerce_variations_page' ) );
            } else {
                add_submenu_page( 'edit.php?post_type=product', __( 'WooCommerce Bulk Variations Manager', self::$text_domain ), __( 'Bulk Variations', self::$text_domain ), 'edit_pages', 'woocommerce_variations', array( $bvm_operation, 'woocommerce_variations_page' ) );
            }

            $welcome_page_name  = __( 'About Bulk Variations Manager', self::$text_domain );
            $welcome_page_title = __( 'Welcome to Bulk Variations Manager', self::$text_domain );

            if ( ! empty( $_GET['page'] ) ) {
                add_submenu_page('edit.php?post_type=product', $welcome_page_name, $welcome_page_title, 'edit_pages', 'bvm-about', array( $GLOBALS['bvm_admin_welcome'], 'show_welcome_page' ) );
                add_submenu_page('edit.php?post_type=product', $welcome_page_name, $welcome_page_title, 'edit_pages', 'bvm-faqs', array( $GLOBALS['bvm_admin_welcome'], 'show_welcome_page' ) );
            }

        }

        function bvm_admin_head() {
            remove_submenu_page( 'edit.php?post_type=product', 'bvm-about' );
            remove_submenu_page( 'edit.php?post_type=product', 'bvm-faqs' );
        }

        function bvm_support_ticket_content() {
            global $pagenow, $typenow;
            
            if ( $pagenow != 'edit.php' ) return;
            
            if ( $typenow != 'product') return;

            if ( ! empty( $_REQUEST['page'] ) && $_REQUEST['page'] != 'bvm-faqs' ) return;

            if ( ! method_exists( 'StoreApps_Upgrade_3_2', 'support_ticket_content' ) ) return;

            $prefix = 'sa_bulk_variations';
            $sku = 'bvm';
            $plugin_data = get_plugin_data( __FILE__ );
            $license_key = get_site_option( $prefix.'_license_key' );
            $text_domain = self::$text_domain;

            StoreApps_Upgrade_3_2::support_ticket_content( $prefix, $sku, $plugin_data, $license_key, $text_domain );
        }

        function add_social_links() {

            if ( ! method_exists( 'StoreApps_Upgrade_3_2', 'add_social_links' ) ) return;

            if ( ( ! empty( $_REQUEST['post_type'] ) && $_REQUEST['post_type'] == 'product' ) && ( ! empty( $_REQUEST['page'] ) && $_REQUEST['page'] == 'woocommerce_variations' ) ) {
                echo '<div class="sa_bulk_variations_social_links" style="padding-bottom: 1em;">' . StoreApps_Upgrade_3_2::add_social_links( 'sa_bulk_variations' ) . '</div>';
            }

        }

        function plugin_action_links( $links ) {
            $action_links = array(
                'about' => '<a href="' . admin_url( 'edit.php?post_type=product&page=bvm-about' ) . '" title="' . esc_attr( __( 'Why Bulk Variations Manager?', self::$text_domain ) ) . '">' . __( 'About', self::$text_domain ) . '</a>',
            );

            return array_merge( $action_links, $links );
        }

        function show_product_count( $category_name = null ) {

            if ( empty( $category_name ) ) return $category_name;
            if ( ! is_admin() ) return $category_name;
            if ( empty( $_GET['post_type'] ) || empty( $_GET['page'] ) ) return $category_name;
            if ( $_GET['post_type'] != 'product' && $_GET['page'] != 'woocommerce_variations' ) return $category_name;

            $is_show_count = get_option( 'bvm_show_category_product_count', 'yes' );

            if ( $is_show_count !== 'yes' ) return $category_name;

            $term = get_term_by( 'name', $category_name, 'product_cat' );

            $category_name .= ' (' . $term->count . ')';

            return $category_name;

        }

        public function ajax_variation_threshold( $threshold = 30, $product = null ) {

            $bvm_threshold = get_option( 'bvm_woocommerce_ajax_variation_threshold', $threshold );

            if ( empty( $bvm_threshold ) ) {
                $bvm_threshold = $threshold;
            }

            return $bvm_threshold;

        }

        function active_plugins_for_quick_help( $active_plugins = array(), $upgrader = null ) {
            global $pagenow, $typenow;
            if ( ( ! empty( $_GET['page'] ) && ( $_GET['page'] == 'bvm-about' || $_GET['page'] == 'bvm-faqs' ) ) 
                    || ( ! empty( $pagenow ) && $pagenow == 'edit.php' && ! empty( $typenow ) && $typenow == 'product' && ! empty( $_GET['page'] ) && $_GET['page'] == 'woocommerce_variations' ) ) {
                $active_plugins['bvm'] = 'bulk-variations-manager';
            } elseif ( array_key_exists( 'bvm', $active_plugins ) ) {
                unset( $active_plugins['bvm'] );
            }
            return $active_plugins;
        }

        /**
         * Function to find all combination
         * 
         * Credit: WooCommerce
         */
        static function array_cartesian( $input ) {
            $result = array();

            @set_time_limit(0);     // prevent timeout

            foreach ( $input as $key => $values ) {
                // If a sub-array is empty, it doesn't affect the cartesian product
                if ( empty( $values ) ) {
                    continue;
                }

                // Special case: seeding the product array with the values from the first sub-array
                if ( empty( $result ) ) {
                    foreach ( $values as $value ) {
                        $result[] = array( $key => $value );
                    }
                }
                else {
                    // Second and subsequent input sub-arrays work like this:
                    //   1. In each existing array inside $product, add an item with
                    //      key == $key and value == first item in input sub-array
                    //   2. Then, for each remaining item in current input sub-array,
                    //      add a copy of each existing array inside $product with
                    //      key == $key and value == first item in current input sub-array

                    // Store all items to be added to $product here; adding them on the spot
                    // inside the foreach will result in an infinite loop
                    $append = array();
                    foreach( $result as &$product ) {
                        // Do step 1 above. array_shift is not the most efficient, but it
                        // allows us to iterate over the rest of the items with a simple
                        // foreach, making the code short and familiar.
                        $product[ $key ] = array_shift( $values );

                        // $product is by reference (that's why the key we added above
                        // will appear in the end result), so make a copy of it here
                        $copy = $product;

                        // Do step 2 above.
                        foreach( $values as $item ) {
                            $copy[ $key ] = $item;
                            $append[] = $copy;
                        }

                        // Undo the side effecst of array_shift
                        array_unshift( $values, $product[ $key ] );
                    }

                    // Out of the foreach, we can add to $results now
                    $result = array_merge( $result, $append );
                }
            }

            return $result;
        }

        static function get_price( $regular_price, $sale_price, $sale_price_dates_from, $sale_price_dates_to ) {
			// Get price if on sale
			if ($sale_price && $sale_price_dates_to == '' && $sale_price_dates_from == '') {
				$price = $sale_price;
			} else { 
				$price = $regular_price;
			}	

			if ($sale_price_dates_from && strtotime($sale_price_dates_from) < strtotime('NOW')) {
				$price = $sale_price;
			}
			
			if ($sale_price_dates_to && strtotime($sale_price_dates_to) < strtotime('NOW')) {
				$price = $regular_price;
			}
			
			return $price;
		}

        /**
         * Find latest StoreApps Upgrade file
         * @return string classname
         */
        function get_latest_upgrade_class() {

            $available_classes = get_declared_classes();
            $available_upgrade_classes = array_filter( $available_classes, function ( $class_name ) {
                                                                                return strpos( $class_name, 'StoreApps_Upgrade_' ) === 0;
                                                                            } );
            $latest_class = 'StoreApps_Upgrade_3_2';
            $latest_version = 0;
            foreach ( $available_upgrade_classes as $class ) {
                $exploded = explode( '_', $class );
                $get_numbers = array_filter( $exploded, function ( $value ) {
                                                            return is_numeric( $value );
                                                        } );
                $version = implode( '.', $get_numbers );
                if ( version_compare( $version, $latest_version, '>' ) ) {
                    $latest_version = $version;
                    $latest_class = $class;
                }
            }

            return $latest_class;
        }

        /**
         * Show notice on admin panel about minimum required version of WooCommerce
         */
        public function needs_wc_25_above() {
            if ( ! $this->is_wc_gte_25() ) {
                $plugin_data = self::get_bvm_plugin_data();
                $plugin_name = $plugin_data['Name'];
                ?>
                <div class="updated error">
                    <p><?php
                        echo sprintf(__( '%1$s %2$s is active but it will only work with WooCommerce 2.5+. %3$s.', self::$text_domain ), '<strong>' . __( 'Important', self::$text_domain ) . ':</strong>', $plugin_name, '<a href="'.admin_url('plugins.php?plugin_status=upgrade').'" target="_blank" >' . __( 'Please update WooCommerce to the latest version', self::$text_domain ) . '</a>' );
                    ?></p>
                </div>
                <?php
            }
        }

        public static function get_bvm_plugin_data() {
            return get_plugin_data( __FILE__ );
        }

	}

}

function initialize_bulk_variations_manager() {
    $GLOBAL['sa_bulk_variations'] = new SA_Bulk_Variations();

    if ( !class_exists( 'StoreApps_Upgrade_3_2' ) ) {
        require_once 'sa-includes/class-storeapps-upgrade-3-2.php';
    }

    $latest_upgrade_class = $GLOBAL['sa_bulk_variations']->get_latest_upgrade_class();

    $sku = 'bvm';
    $prefix = 'sa_bulk_variations';
    $plugin_name = __( 'WooCommerce Bulk Variations Manager', SA_Bulk_Variations::$text_domain );
    $documentation_link = 'https://www.storeapps.org/knowledgebase_category/bulk-variations-manager/';
    $GLOBALS['bulk_variations_manager_upgrade'] = new $latest_upgrade_class( __FILE__, $sku, $prefix, $plugin_name, SA_Bulk_Variations::$text_domain, $documentation_link );

    if ( ! class_exists( 'SA_In_App_Offer' ) ) {
        include_once 'sa-includes/class-sa-in-app-offer.php';
        $args = array(
            'file'           => __FILE__,
            'prefix'         => 'bvm',
            'option_name'    => 'sa_offer_halloween_2018',
            'campaign'       => 'sa_halloween_2018',
            'start'          => '2018-10-30',
            'end'            => '2018-11-02',
            'is_plugin_page' => $GLOBAL['sa_bulk_variations']->is_page_for_notifications(),
        );
        SA_In_App_Offer::get_instance( $args );
    }

}
add_action( 'plugins_loaded', 'initialize_bulk_variations_manager' );
